#!/usr/bin/python2
# -*- coding: utf-8 -*-

__author__ = 'Patryk Stachurski <patryk.stachurski@open-e.com>'
__revision__ = '$Id'

from jovianapi.resource.core import Resource
from jovianapi.resource.core import ResourcesContainer
from jovianapi.resource.core import UNDEFINED


class ShareModel(Resource):
    class NfsModel(Resource):
        FIELDS_NAMES = ['no_root_squash', 'allow_write_ip', 'all_squash', 'enabled', 'insecure_connections',
                        'synchronous_data_record', 'insecure_lock_requests', 'allow_access_ip']

        def __init__(self, no_root_squash=UNDEFINED, allow_write_ip=UNDEFINED, all_squash=UNDEFINED, enabled=UNDEFINED,
                     insecure_connections=UNDEFINED, synchronous_data_record=UNDEFINED,
                     insecure_lock_requests=UNDEFINED, allow_access_ip=UNDEFINED, api=None):
            """
            :type no_root_squash: bool
            :type allow_write_ip: list
            :type all_squash: bool
            :type enabled: bool
            :type insecure_connections: bool
            :type synchronous_data_record: bool
            :type insecure_lock_requests: bool
            :type allow_access_ip: list
            :type api: jovianapi.API
            """
            Resource.__init__(self, api=api)

            self.no_root_squash = no_root_squash
            self.allow_write_ip = allow_write_ip
            self.all_squash = all_squash
            self.enabled = enabled
            self.insecure_connections = insecure_connections
            self.synchronous_data_record = synchronous_data_record
            self.insecure_lock_requests = insecure_lock_requests
            self.allow_access_ip = allow_access_ip

    class SmbModel(Resource):
        FIELDS_NAMES = ['read_only', 'enabled', 'visible', 'default_case', 'inherit_perms', 'handling_large_dirs',
                        'inherit_owner', 'access_mode', 'map_acl_inherit']

        def __init__(self, read_only=UNDEFINED, enabled=UNDEFINED, visible=UNDEFINED, default_case=UNDEFINED,
                     inherit_perms=UNDEFINED, handling_large_dirs=UNDEFINED, inherit_owner=UNDEFINED,
                     access_mode=UNDEFINED, map_acl_inherit=UNDEFINED, api=None):
            """
            :type read_only: bool
            :type enabled: bool
            :type visible: bool
            :type default_case: unicode
            :type inherit_perms: bool
            :type handling_large_dirs: bool
            :type inherit_owner: bool
            :type access_mode: unicode
            :type map_acl_inherit: bool
            :type api: jovianapi.API
            """
            Resource.__init__(self, api=api)

            self.read_only = read_only
            self.enabled = enabled
            self.visible = visible
            self.default_case = default_case
            self.inherit_perms = inherit_perms
            self.handling_large_dirs = handling_large_dirs
            self.inherit_owner = inherit_owner
            self.access_mode = access_mode
            self.map_acl_inherit = map_acl_inherit

    FIELDS_NAMES = ['comment', 'name', 'real_path', 'active', 'path', 'conflicted', 'nfs', 'smb',
                    'missing_configuration', 'missing_directory']

    def __init__(self, comment=UNDEFINED, name=UNDEFINED, real_path=UNDEFINED, active=UNDEFINED, path=UNDEFINED,
                 conflicted=UNDEFINED, nfs=UNDEFINED, smb=UNDEFINED, missing_configuration=UNDEFINED,
                 missing_directory=UNDEFINED, dataset=None, api=None):
        """
        :type comment: unicode
        :type name: unicode
        :type real_path: unicode
        :type nfs: NfsModel
        :type active: bool
        :type path: unicode
        :type conflicted: bool
        :type smb: SmbModel
        :type missing_configuration: bool
        :type missing_directory: bool
        :type dataset: jovianapi.resource.dataset.Dataset
        :type api: jovianapi.API
        """
        Resource.__init__(self, api=api)

        self.comment = comment
        self.name = name
        self.real_path = real_path
        self.active = active
        self.path = path
        self.conflicted = conflicted
        self.nfs = self.NfsModel.from_dict(nfs)
        self.smb = self.SmbModel.from_dict(smb)

        self.dataset = dataset



class Share(ShareModel):
    """
    Example JSON::

        {
            "comment": "",
            "name": "MyShare",
            "real_path": "/Pools/Pool-0/MyDataSet/MyShareDir",
            "nfs":
            {
                "no_root_squash": false,
                "allow_write_ip":
                [
                ],
                "all_squash": false,
                "enabled": false,
                "insecure_connections": false,
                "synchronous_data_record": false,
                "insecure_lock_requests": false,
                "allow_access_ip":
                [
                ]
            },
            "active": true,
            "path": "Pool-0/MyDataSet/MyShareDir",
            "conflicted": false
        }
    """


class SharesContainer(ResourcesContainer):
    def __init__(self, dataset):
        """
        :type pool: jovianapi.resource.dataset.Dataset
        """
        self.dataset = dataset

        super(SharesContainer, self).__init__(dataset.api)

    def fetch(self):
        items = self.driver.list_shares().data.get('entries')  # REST returns output in different format
        all_shares = [Share.from_dict(i, dataset=self.dataset, api=self.api) for i in items]
        return filter(lambda s: s.path.split('/')[1] == self.dataset.name, all_shares)  # pylint: disable=bad-builtin

    def create(self, name, nfs_options=None, smb_options=None):
        """Creates share
        """
        # Add prefix automatically
        path = '{self.dataset.pool.name}/{self.dataset.name}/{name}'.format(self=self, name=name)

        data = dict(name=name, path=path)

        if nfs_options is not None:
            data['nfs'] = nfs_options
        if smb_options is not None:
            data['smb'] = smb_options

        self.driver.create_share(data)
        self.fetch()
        return self[name]

    def __getitem__(self, key):
        """
        :rtype: Share
        """
        if isinstance(key, basestring):
            if '/' not in key:
                path = '{self.dataset.pool.name}/{self.dataset.name}/{name}'.format(self=self, name=key)

            for item in self:
                value = getattr(item, 'path', None)

                if value == path:
                    return item

        return super(SharesContainer, self).__getitem__(key)

