#!/usr/bin/python2
# -*- coding: utf-8 -*-

__author__ = 'Patryk Stachurski <patryk.stachurski@open-e.com>'
__revision__ = '$Id'

import sys
import json


class UNDEFINED(type):
    class __metaclass__(type):
        def __repr__(cls):
            return 'UNDEFINED'

    def __nonzero__(cls):
        return False


class ResourceModel(object):
    """Resource is the most basic type of API elements.

    Each resource is described as :class:`dict`.

    This class is responsible for parsing and representing data returned by API driver.

    Most of the time you won't need to create resource manually.
    """
    FIELDS_NAMES = []

    @classmethod
    def from_dict(cls, data_dict, api=None, **kwargs):
        if isinstance(data_dict, ResourceModel):
            data_dict = data_dict.to_dict()

        # Remove unsupported fields
        for key in data_dict.keys():
            if key not in cls.FIELDS_NAMES:
                del data_dict[key]
                msg = "WARNING: resource {cls.__name__!r} doesn't define field {key!r}\n".format(cls=cls, key=key)
                sys.stderr.write(msg)

        if data_dict is UNDEFINED:
            data_dict = dict()

        if api is not None:
            data_dict['api'] = api

        data_dict.update(kwargs)
        return cls(**data_dict)

    @classmethod
    def from_json(cls, json_string):
        return cls.from_dict(json.loads(json_string))

    def to_dict(self):
        data_dict = dict()
        for field_name in self.FIELDS_NAMES:
            field_value = getattr(self, field_name)

            if field_value is UNDEFINED:
                continue

            if isinstance(field_value, ResourceModel):
                field_model = field_value.to_dict()
                if field_model:
                    data_dict[field_name] = field_model

            elif isinstance(field_value, list):
                data_dict[field_name] = []

                for item in field_value:
                    if isinstance(item, ResourceModel):
                        item_model = item.to_dict()
                        if item_model:
                            data_dict[field_name].append(item_model)
                    else:
                        data_dict[field_name].append(item)

            else:
                data_dict[field_name] = field_value

        return data_dict

    def to_json(self, indent=4):
        return json.dumps(self.to_dict(), indent=indent)

    def __repr__(self):
        attrs = []
        if 'id' in self.FIELDS_NAMES:
            attrs.append('id')
        else:
            for field in self.FIELDS_NAMES:
                if 'id' in field:
                    attrs.append(field)
                break

        if 'name' in self.FIELDS_NAMES:
            attrs.append('name')
        else:
            for field in self.FIELDS_NAMES:
                if 'name' in field:
                    attrs.append(field)
                break

        attrs_def = ', '.join('%s=%r' % (n, getattr(self, n)) for n in attrs)
        return '{self.__class__.__name__}({attrs_def})'.format(self=self, attrs_def=attrs_def)


class Resource(ResourceModel):
    def __init__(self, api=None):
        """
        :type api: jovianapi.API
        """
        self.api = api

    @property
    def driver(self):
        """
        :rtype: jovianapi.driver.rest.DriverREST
        """
        return self.api.driver

    def fetch(self):
        raise NotImplementedError()

    def apply(self):
        raise NotImplementedError()


class ResourcesContainer(list):
    """Resources container

    """
    def __init__(self, api):
        self.api = api

        items = self.fetch()
        list.__init__(self, items)

    @property
    def driver(self):
        """
        :rtype: jovianapi.driver.rest.DriverREST
        """
        return self.api.driver

    def fetch(self):
        raise NotImplementedError()

    def refresh(self):
        items = self.fetch()
        while self:
            self.pop()
        self.extend(items)

    def __getitem__(self, key):
        self.refresh()

        if isinstance(key, int):
            return list.__getitem__(self, key)

        for item in self:
            item_name = getattr(item, 'name', None)

            if item_name == key:
                return item

        raise KeyError(key)
