#!/usr/bin/python2
# -*- coding: utf-8 -*-

__author__ = 'Patryk Stachurski <patryk.stachurski@open-e.com>'
__revision__ = '$Id'

import warnings
from time import sleep

import requests
from requests.auth import HTTPBasicAuth
from requests.packages.urllib3.exceptions import InsecureRequestWarning

from jovianapi.driver.abstract import AbstractDriver
from jovianapi.driver.rest.response import Response


# Hide warnings about unverified HTTPS
warnings.simplefilter(action='ignore', category=InsecureRequestWarning)


class DriverException(Exception):
    def __init__(self, message, code, cls_name, endpoint):
        self.message = message
        self.code = code
        self.cls_name = cls_name
        self.endpoint = endpoint

        Exception.__init__(self, message)

    def __str__(self):
        return '(HTTP {self.code} {self.endpoint}) {self.cls_name}: {self.message}'.format(self=self)


class PostException(DriverException):
    pass


class DriverREST(AbstractDriver):
    DEFAULT_VERIFY_CERT = False

    GET_TIMEOUT = 30

    def __init__(self, address, port, username, password, verify_cert=DEFAULT_VERIFY_CERT, secure=None):
        self.address = address
        self.port = port
        self.username = username
        self.password = password

        self.verify_cert = verify_cert

        if secure is not None:
            self.secure = secure
        elif not username:
            self.secure = False
        else:
            self.secure = True

    def create_url(self, endpoint, api_version=3):
        if endpoint.startswith('/'):
            endpoint = endpoint[1:]

        if self.secure:
            protocol = 'https'
        else:
            protocol = 'http'

        return '{protocol}://{self.address}:{self.port}/api/v{api_version}/{endpoint}'.format(
                protocol=protocol, self=self, api_version=api_version, endpoint=endpoint)

    def create_auth(self):
        if self.secure:
            return HTTPBasicAuth(self.username, self.password)
        else:
            return None

    ###
    # HTTP METHODS IMPLEMENTATION
    def get(self, endpoint, timeout=GET_TIMEOUT):
        url = self.create_url(endpoint)
        auth = self.create_auth()

        response = requests.get(url, auth=auth, verify=self.verify_cert, timeout=timeout)

        rest_response = Response(response.json(), response.status_code)
        if rest_response.code != 200:
            raise DriverException(rest_response.error.message, rest_response.code,
                                  rest_response.error.cls_name, endpoint)

        return rest_response

    def post(self, endpoint, data, parse_response=True):
        url = self.create_url(endpoint)
        auth = self.create_auth()

        output = requests.post(url, auth=auth, verify=self.verify_cert, json=data, headers={'Content-Type': 'application/json'})

        if not parse_response:
            return output.content

        try:
            output_json = output.json()
        except ValueError as e:
            msg = e.message + '\n\n%s' % output.content
            raise ValueError(msg)

        response = Response(output_json, output.status_code)

        if response.error:
            exception = DriverException(response.error.message, response.code, response.error.cls_name, endpoint)
            raise exception

        # FIXME: rest doesn't know yet about created object
        sleep(2)

        return response

    def delete(self, endpoint, data=None):
        url = self.create_url(endpoint)
        auth = self.create_auth()

        data = data or dict()

        output = requests.delete(url, auth=auth, verify=self.verify_cert, json=data, headers={'Content-Type': 'application/json'})

        try:
            output_json = output.json()
        except ValueError:
            output_json = dict()

        response = Response(output_json, output.status_code)

        if response.error:
            exception = DriverException(response.error.message, response.code, response.error.cls_name, endpoint)
            raise exception

        return response

    def put(self, endpoint, data, parse_response=True):
        url = self.create_url(endpoint)
        auth = self.create_auth()

        output = requests.put(url, auth=auth, verify=self.verify_cert, json=data, headers={'Content-Type': 'application/json'})

        if not parse_response:
            return output.content

        try:
            output_json = output.json()
        except ValueError as e:
            msg = e.message + '\n\n%s' % output.content
            raise ValueError(msg)

        response = Response(output_json, output.status_code)

        if response.error:
            exception = DriverException(response.error.message, response.code, response.error.cls_name, endpoint)
            raise exception

        # FIXME: rest doesn't know yet about created object
        sleep(1)

        return response

    def disks_rescan(self):
        return self.post('/disks/rescan', dict(), parse_response=False)

    def get_disk(self, disk_id):
        return self.get('/disk/{id}'.format(id=disk_id))

    def list_disks(self):
        return self.get('/disks')

    def list_disks_used(self):
        return self.get('/disks/used')

    def list_disks_unused(self):
        return self.get('/disks/unused')

    def list_pools(self):
        return self.get('/pools')

    def list_pools_exported(self):
        return self.get('/pools/import')

    def create_pool(self, data):
        return self.post('/pools', data)

    def destroy_exported_pool(self):
        return self.delete

    def import_pool(self, data):
        endpoint = '/pools/import'
        return self.post(endpoint, data)

    def export_pool(self, data):
        endpoint = '/pools/{name}/export'.format(name=data['name'])
        return self.post(endpoint, data)

    def list_datasets(self, pool_name):
        endpoint = '/pools/{pool_name}/nas-volumes'.format(pool_name=pool_name)
        return self.get(endpoint)

    def create_dataset(self, pool_name, data):
        endpoint = '/pools/{pool_name}/nas-volumes'.format(pool_name=pool_name)
        return self.post(endpoint, data)

    def list_shares(self):
        return self.get('/shares')

    def create_share(self, data):
        return self.post('/shares', data)

    def modify_share(self, data):
        endpoint = '/shares/{name}'.format(name=data['name'])
        return self.put(endpoint, data)

    def list_targets(self, pool_name):
        endpoint = '/pools/{pool_name}/san/iscsi/targets'.format(pool_name=pool_name)
        return self.get(endpoint)

    def create_target(self, pool_name, data):
        endpoint = '/pools/{pool_name}/san/iscsi/targets'.format(pool_name=pool_name)
        return self.post(endpoint, data)

    def list_volumes(self, pool_name):
        endpoint = '/pools/{pool_name}/volumes'.format(pool_name=pool_name)
        return self.get(endpoint)

    def create_volume(self, pool_name, data):
        endpoint = '/pools/{pool_name}/volumes'.format(pool_name=pool_name)
        return self.post(endpoint, data)

    def attach_lun_to_target(self, pool_name, target_name, data):
        endpoint = '/pools/{pool_name}/san/iscsi/targets/{target_name}/luns'.format(
                pool_name=pool_name, target_name=target_name)
        return self.post(endpoint, data)

    def list_nics(self):
        return self.get('/network/interfaces')

    def list_cluster_nodes(self):
        return self.get('/cluster/nodes')

    def list_cluster_zfs_resources(self):
        return self.get('/cluster/resources')

    def move_cluster_resource(self, resource_name, data):
        endpoint = '/cluster/resources/{resource_name}/move-resource'.format(resource_name=resource_name)
        return self.post(endpoint, data)

